$(document).ready(function() {
    // Formateador de MAC address
    function formatMacInput(input) {
        input.addEventListener('input', function(e) {
            let value = e.target.value.replace(/[^a-fA-F0-9]/g, "").toUpperCase();
            let formatted = "";
            for (let i = 0; i < value.length && i < 12; i++) {
                formatted += value[i];
                if ((i + 1) % 2 === 0 && i < value.length - 1) {
                    formatted += ":";
                }
            }
            e.target.value = formatted;
        });
    }

    // Mostrar mensajes
    function showMessage(type, message) {
        const alertTypes = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'info': 'alert-info'
        };
        
        const alertClass = alertTypes[type] || 'alert-info';
        const messageHtml = `
            <div class="alert ${alertClass} alert-dismissible fade show">
                ${message}
                <button type="button" class="close" data-dismiss="alert">&times;</button>
            </div>
        `;
        
        $('#message-container').html(messageHtml);
        
        // Auto cerrar mensajes después de 5 segundos
        setTimeout(() => {
            $('.alert').alert('close');
        }, 5000);
    }

    // Cargar resultados de búsqueda
    function loadSearchResults(searchTerm) {
        $.get('api/buscar_clientes.php', { action: 'search', search: searchTerm }, function(response) {
            if (response.success) {
                renderResults(response.results, response.dnsMap);
            } else {
                showMessage('error', response.error);
            }
        }).fail(function() {
            showMessage('error', 'Error al realizar la búsqueda');
        });
    }

    // Renderizar resultados
    function renderResults(results, dnsMap) {
        if (results.length === 0) {
            $('#search-results').html(`
                <div class="alert alert-info">
                    No se encontraron clientes con ese criterio de búsqueda
                </div>
            `);
            return;
        }

        let html = `
            <div class="table-responsive">
                <table class="table table-bordered">
                    <thead class="thead-dark">
                        <tr>
                            <th>Cliente</th>
                            <th>MAC</th>
                            <th>Usuario</th>
                            <th>DNS</th>
                            <th>Vencimiento</th>
                            <th>Estado</th>
                            <th>Protección</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
        `;

        results.forEach(row => {
            const today = new Date();
const [year, month, day] = row.expiracion.split('-');
const expiration = new Date(year, month - 1, day); // Mes base 0
const daysLeft = Math.floor((expiration - today) / (1000 * 60 * 60 * 24));
;
            
            let status, statusClass;
            if (daysLeft > 3) {
                status = 'Activo';
                statusClass = 'success';
            } else if (daysLeft >= 0) {
                status = `Por vencer (${daysLeft} días)`;
                statusClass = 'warning';
            } else {
                status = 'Vencido';
                statusClass = 'danger';
            }

            const formattedDate = expiration.toLocaleDateString('es-ES');
            
            // Botón WhatsApp solo para cuentas por vencer (entre 0 y 3 días)
            const whatsappBtn = (daysLeft <= 3 && daysLeft >= 0) ? `
                <button class="btn btn-sm whatsapp-reminder" 
                        onclick="sendReminder(${row.id}, ${daysLeft})"
                        title="Enviar recordatorio por WhatsApp">
                    <i class="fab fa-whatsapp"></i> Recordatorio
                </button>
            ` : '';
            
            html += `
                <tr>
                    <td>${escapeHtml(row.cliente_nombre)}</td>
                    <td>${escapeHtml(row.mac_address)}</td>
                    <td>${escapeHtml(row.username)}</td>
                    <td>${escapeHtml(dnsMap[row.dns_id] || 'N/A')}</td>
                    <td>${formattedDate}</td>
                    <td><span class="badge badge-${statusClass}">${status}</span></td>
                    <td class="text-center">
                        <button class="btn btn-sm protection-btn ${row.is_protected ? 'btn-success' : 'btn-secondary'}" 
                                onclick="toggleProtection(${row.id}, ${row.is_protected})"
                                title="${row.is_protected ? 'Protegido - Click para desproteger' : 'Editable - Click para proteger'}">
                            <i class="fas ${row.is_protected ? 'fa-lock' : 'fa-lock-open'}"></i>
                            ${row.is_protected ? 'Protegido' : 'Libre'}
                        </button>
                    </td>
                    <td class="text-center">
                        <button class="btn btn-sm btn-primary" onclick="loadEditForm(${row.id})">
                            <i class="fas fa-edit"></i> Editar
                        </button>
                        
                        <button class="btn btn-sm btn-danger" onclick="confirmDelete(${row.id})">
                            <i class="fas fa-trash-alt"></i> Eliminar
                        </button>
                        
                        ${whatsappBtn}
                    </td>
                </tr>
            `;
        });

        html += `
                    </tbody>
                </table>
            </div>
        `;

        $('#search-results').html(html);
    }

    // Escapar HTML
    function escapeHtml(unsafe) {
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }

    // Cambiar protección
    window.toggleProtection = function(id, currentStatus) {
        $.get('api/buscar_clientes.php', { action: 'toggle_protect', id: id }, function(response) {
            if (response.success) {
                showMessage('success', response.message);
                // Recargar resultados
                loadSearchResults($('#search-input').val());
            } else {
                showMessage('error', response.error);
            }
        }).fail(function() {
            showMessage('error', 'Error al cambiar el estado de protección');
        });
    };

    // Cargar formulario de edición
   window.loadEditForm = function(id) {
    $.get('api/buscar_clientes.php', { action: 'get_client', id: id }, function(response) {
        if (response.success) {
            const client = response.client;
            
            // Corregir la fecha para evitar problemas de zona horaria
            const expiracionDate = new Date(client.expiracion);
            const correctedDate = new Date(expiracionDate.getTime() + Math.abs(expiracionDate.getTimezoneOffset() * 60000));
            const formattedDate = correctedDate.toISOString().split('T')[0];
            
            const dnsOptions = Object.entries(response.dnsMap).map(([id, title]) => `
                <option value="${id}" ${id == client.dns_id ? 'selected' : ''}>${escapeHtml(title)}</option>
            `).join('');
            
            const editFormHtml = `
                    <input type="hidden" name="id" value="${client.id}">
                    <input type="hidden" name="action" value="update_client">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label><i class="fas fa-network-wired"></i> MAC del dispositivo</label>
                                <input type="text" name="mac_address" class="form-control mac-input" 
                                       value="${escapeHtml(client.mac_address)}" required
                                       pattern="^([0-9A-Fa-f]{2}[:-]){2,7}[0-9A-Fa-f]{2}$"
                                       title="Formato MAC: 00:1A:2B:3C:4D:5E">
                            </div>
                            
                            <div class="form-group">
                                <label><i class="fas fa-user"></i> Usuario</label>
                                <input type="text" name="username" class="form-control" 
                                       value="${escapeHtml(client.username)}" required>
                            </div>
                            
                            <div class="form-group">
                                <label><i class="fas fa-key"></i> Contraseña</label>
                                <input type="text" name="password" class="form-control" 
                                       value="${escapeHtml(client.password)}" required>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="form-group">
                                <label><i class="fas fa-server"></i> DNS</label>
                                <select class="form-control" name="dns_id" required>
                                    <option value="">Seleccione DNS...</option>
                                    ${dnsOptions}
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label><i class="fas fa-user-tag"></i> Nombre del cliente</label>
                                <input type="text" name="cliente_nombre" class="form-control" 
                                       value="${escapeHtml(client.cliente_nombre)}" required>
                            </div>
                            
                            <div class="form-group">
                                <label><i class="fab fa-whatsapp"></i> WhatsApp (con código de país)</label>
                                <input type="text" name="cliente_whatsapp" class="form-control" 
                                       value="${escapeHtml(client.cliente_whatsapp)}" 
                                       placeholder="593980649398" required
                                       pattern="^\\+?\\d{1,3}\\d{6,14}$"
                                       title="Ingrese número con código de país (ej: 593980649398)">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                    <label><i class="fas fa-calendar-alt"></i> Fecha de vencimiento</label>
                    <input type="date" name="expiracion" class="form-control" 
                           value="${formattedDate}" 
                           min="${new Date().toISOString().split('T')[0]}" required>
                </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group form-check pt-4">
                                <input type="checkbox" class="form-check-input" name="is_protected" 
                                       id="is_protected" value="1" ${client.is_protected ? 'checked' : ''}>
                                <label class="form-check-label" for="is_protected">
                                    <i class="fas fa-lock"></i> Proteger lista (evitar modificaciones en la app)
                                </label>
                            </div>
                        </div>
                    </div>
                `;
                
                $('#edit-modal-body').html(editFormHtml);
            formatMacInput(document.querySelector('#editModal .mac-input'));
            $('#editModal').modal('show');
            } else {
                showMessage('error', response.error);
            }
        }).fail(function() {
            showMessage('error', 'Error al cargar los datos del cliente');
        });
    };

    // Confirmar eliminación
    window.confirmDelete = function(id) {
        $('#confirm-delete-btn').off('click').on('click', function() {
            deleteClient(id);
        });
        $('#confirmModal').modal('show');
    };

    // Eliminar cliente
    window.deleteClient = function(id) {
        $.post('api/buscar_clientes.php', { action: 'delete_client', id: id }, function(response) {
            $('#confirmModal').modal('hide');
            if (response.success) {
                showMessage('success', response.message);
                loadSearchResults($('#search-input').val());
            } else {
                showMessage('error', response.error);
            }
        }).fail(function() {
            $('#confirmModal').modal('hide');
            showMessage('error', 'Error al eliminar el cliente');
        });
    };

    // Enviar recordatorio por WhatsApp
    window.sendReminder = function(id, daysLeft) {
        $.get('api/buscar_clientes.php', { action: 'get_client', id: id }, function(response) {
            if (response.success) {
                const client = response.client;
                const message = `
                    ¡Hola ${client.cliente_nombre}! 👋\n\n
                    📢 *Recordatorio importante:*\n
                    Tu servicio está por vencer en *${daysLeft} día(s)*.\n
                    📅 *Fecha de vencimiento:* ${new Date(client.expiracion).toLocaleDateString('es-ES')}\n\n
                    Por favor, realiza tu renovación para evitar interrupciones. ¡Gracias! 😊\n
                    ¿Necesitas ayuda? Estamos aquí para asistirte. 📞
                `;
                
                window.open(`https://wa.me/${client.cliente_whatsapp}?text=${encodeURIComponent(message)}`, '_blank');
            } else {
                showMessage('error', response.error);
            }
        });
    };

    // Enviar formulario de búsqueda
    $('#search-form').submit(function(e) {
        e.preventDefault();
        const searchTerm = $('#search-input').val().trim();
        if (searchTerm) {
            loadSearchResults(searchTerm);
        }
    });

    // Enviar formulario de edición
    $('#edit-form').submit(function(e) {
        e.preventDefault();
        const formData = $(this).serialize();
        
        $.post('api/buscar_clientes.php', formData, function(response) {
            if (response.success) {
                showMessage('success', response.message);
                $('#editModal').modal('hide');
                loadSearchResults($('#search-input').val());
            } else {
                showMessage('error', response.error);
            }
        }).fail(function() {
            showMessage('error', 'Error al actualizar el cliente');
        });
    });

    // Cargar resultados si hay término de búsqueda en la URL
    const urlParams = new URLSearchParams(window.location.search);
    const searchParam = urlParams.get('search');
    if (searchParam) {
        $('#search-input').val(searchParam);
        loadSearchResults(searchParam);
    }
});